/**
 * Orthanc - A Lightweight, RESTful DICOM Store
 *
 * Copyright (C) 2012-2016 Sebastien Jodogne <s.jodogne@orthanc-labs.com>,
 * Medical Physics Department, University Hospital of Liege, Belgium
 * Copyright (C) 2017-2023 Osimis S.A., Belgium
 * Copyright (C) 2024-2025 Orthanc Team SRL, Belgium
 * Copyright (C) 2021-2025 Sebastien Jodogne, ICTEAM UCLouvain, Belgium
 *
 * Copyright (c) 2012 The Chromium Authors. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are
 * met:
 *
 *    * Redistributions of source code must retain the above copyright
 * notice, this list of conditions and the following disclaimer.
 *    * Redistributions in binary form must reproduce the above
 * copyright notice, this list of conditions and the following disclaimer
 * in the documentation and/or other materials provided with the
 * distribution.
 *    * Neither the name of Google Inc., the name of the University Hospital of Liege,
 * nor the names of its contributors may be used to endorse or promote
 * products derived from this software without specific prior written
 * permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 **/


#pragma once


#if ORTHANC_ENABLE_SQLITE != 1
#  error Macro ORTHANC_ENABLE_SQLITE must be set to 1 to use SQLite
#endif


#if ORTHANC_SQLITE_STANDALONE == 1
#include <stdexcept>

namespace Orthanc
{
  namespace SQLite
  {
    // Auto-generated by "Resources/CodeGeneration/GenerateErrorCodes.py"
    enum ErrorCode
    {
      ErrorCode_ParameterOutOfRange,
      ErrorCode_BadParameterType,
      ErrorCode_SQLiteNotOpened,
      ErrorCode_SQLiteAlreadyOpened,
      ErrorCode_SQLiteCannotOpen,
      ErrorCode_SQLiteStatementAlreadyUsed,
      ErrorCode_SQLiteExecute,
      ErrorCode_SQLiteRollbackWithoutTransaction,
      ErrorCode_SQLiteCommitWithoutTransaction,
      ErrorCode_SQLiteRegisterFunction,
      ErrorCode_SQLiteFlush,
      ErrorCode_SQLiteCannotRun,
      ErrorCode_SQLiteCannotStep,
      ErrorCode_SQLiteBindOutOfRange,
      ErrorCode_SQLitePrepareStatement,
      ErrorCode_SQLiteTransactionAlreadyStarted,
      ErrorCode_SQLiteTransactionCommit,
      ErrorCode_SQLiteTransactionBegin
    };

    class OrthancSQLiteException : public ::std::runtime_error
    {
    public:
      OrthancSQLiteException(ErrorCode error) :
        ::std::runtime_error(EnumerationToString(error))
      {
      }

      // Auto-generated by "Resources/CodeGeneration/GenerateErrorCodes.py"
      static const char* EnumerationToString(ErrorCode code)
      {
        switch (code)
        {
          case ErrorCode_ParameterOutOfRange:
            return "Parameter out of range";

          case ErrorCode_BadParameterType:
            return "Bad type for a parameter";

          case ErrorCode_SQLiteNotOpened:
            return "SQLite: The database is not opened";

          case ErrorCode_SQLiteAlreadyOpened:
            return "SQLite: Connection is already open";

          case ErrorCode_SQLiteCannotOpen:
            return "SQLite: Unable to open the database";

          case ErrorCode_SQLiteStatementAlreadyUsed:
            return "SQLite: This cached statement is already being referred to";

          case ErrorCode_SQLiteExecute:
            return "SQLite: Cannot execute a command";

          case ErrorCode_SQLiteRollbackWithoutTransaction:
            return "SQLite: Rolling back a nonexistent transaction (have you called Begin()?)";

          case ErrorCode_SQLiteCommitWithoutTransaction:
            return "SQLite: Committing a nonexistent transaction";

          case ErrorCode_SQLiteRegisterFunction:
            return "SQLite: Unable to register a function";

          case ErrorCode_SQLiteFlush:
            return "SQLite: Unable to flush the database";

          case ErrorCode_SQLiteCannotRun:
            return "SQLite: Cannot run a cached statement";

          case ErrorCode_SQLiteCannotStep:
            return "SQLite: Cannot step over a cached statement";

          case ErrorCode_SQLiteBindOutOfRange:
            return "SQLite: Bind a value while out of range (serious error)";

          case ErrorCode_SQLitePrepareStatement:
            return "SQLite: Cannot prepare a cached statement";

          case ErrorCode_SQLiteTransactionAlreadyStarted:
            return "SQLite: Beginning the same transaction twice";

          case ErrorCode_SQLiteTransactionCommit:
            return "SQLite: Failure when committing the transaction";

          case ErrorCode_SQLiteTransactionBegin:
            return "SQLite: Cannot start a transaction";

          default:
            return "Unknown error code";
        }
      }
    };
  }
}

#else
#  include "../OrthancException.h"
#  define OrthancSQLiteException ::Orthanc::OrthancException
#endif
